SuperStrict

Rem
bbdoc: Data structures/Binary search trees

Module pine.BinTree

ModuleInfo "Version: 1.03"
ModuleInfo "Author: Sophie Kirschner : meapineapple@gmail.com"
ModuleInfo "License: Creative Commons Attribution-ShareAlike 3.0 Unported (CC BY-SA 3.0)"
ModuleInfo "Modserver: pine"

ModuleInfo "History: 1.04 Release"
ModuleInfo "History: Rennamed BinNode.removevalue to BinNode.removeonevalue, changed behavior of BinNode.removevalue"
ModuleInfo "History: Added BinNode.containsvalue()"
ModuleInfo "History: Added TreeRemoveValue()"
ModuleInfo "History: 1.03 Release"
ModuleInfo "History: Added TreeFindCount"
ModuleInfo "History: 1.02 Release"
ModuleInfo "History: Added BinNodeValue"
ModuleInfo "History: 1.01 Release"
ModuleInfo "History: Added splay tree functionality"
ModuleInfo "History: changed BinTree.remove() to return a node instead of a boolean and added TreeRemoveFirst"
ModuleInfo "History: Revised handling of duplicate keys"

End Rem


Import brl.math
Import brl.linkedlist


	Rem
	bbdoc: Create a new tree
	returns: A new tree
	End Rem
Function CreateTree:BinTree(splay:Int=True)
	Return BinTree.Create(splay)
End Function
	Rem
	bbdoc: Set whether a tree will automatically splay when not otherwise specified
	End Rem
Function TreeSetSplay(tree:BinTree,splay:Int)
	tree.autosplay=splay<>0
End Function
	Rem
	bbdoc: Get whether a tree will automatically splay when not otherwise specified
	returns: 1 if the tree is set to splay automatically, 0 if not
	End Rem
Function TreeGetSplay:Int(tree:BinTree)
	Return tree.autosplay
End Function
	Rem
	bbdoc: Get the root node of a tree
	returns: The root node of the tree
	End Rem
Function TreeRoot:BinNode(tree:BinTree)
	Return tree.root
End Function
	Rem
	bbdoc: Splay a node so that it becomes the new root of a tree
	End Rem
Function TreeSplayNode(tree:BinTree,node:BinNode)
	tree.splay node
End Function
	Rem
	bbdoc: Clear all contents of a tree
	End Rem
Function ClearTree(tree:BinTree)
	tree.clear
End Function
	Rem
	bbdoc: Count the number of values in a tree
	returns: The number of values in the tree
	End Rem
Function CountTree:Int(tree:BinTree)
	Return tree.countvalues()
End Function
	Rem
	bbdoc: Count the number of nodes in a tree
	returns: The number of nodes in the tree
	End Rem
Function CountTreeNodes:Int(tree:BinTree)
	Return tree.count()
End Function
	Rem
	bbdoc: A tree with no nodes is considered to have a height of 0, and a tree with only a root to have a height of 1.
	returns: The height of the tree
	End Rem
Function TreeHeight:Int(tree:BinTree)
	Return tree.height()
End Function
	Rem
	bbdoc: Determine whether the tree is empty
	returns: 1 if the tree is empty, 0 if it contains at least a root
	End Rem
Function TreeIsEmpty:Int(tree:BinTree)
	Return tree.isempty()
End Function
	Rem
	bbdoc: Determine whether some key is present in the tree
	returns: 1 if there is any node in the tree with the specified key, 0 if there is not
	End Rem
Function TreeContains:Int(tree:BinTree,key:String)
	Return tree.contains(key)
End Function
	Rem
	bbdoc: Insert a new key and value pair into the tree
	returns: The newly inserted node
	End Rem
Function TreeInsert:BinNode(tree:BinTree,key:String,value:Object)
	Return tree.insert(key,value)
End Function
	Rem
	bbdoc: Remove the all values associated with the desired key
	returns: The number of values removed
	End Rem
Function TreeRemove:Int(tree:BinTree,key:String)
	Local n:BinNode=tree.removeall(key)
	If n Return n.valuecount()
	Return 0
End Function
	Rem
	bbdoc: Remove the first value with the desired key
	returns: The node the value was removed from, or null if nothing was found to be removed
	End Rem
Function TreeRemoveFirst:BinNode(tree:BinTree,key:String)
	Return tree.remove(key)
End Function
	Rem
	bbdoc: Remove a specific node.
	End Rem
Function TreeRemoveNode(tree:BinTree,node:BinNode)
	tree.removenode node
End Function
	Rem
	bbdoc: Remove a specific value.
	about: If a key is specified then it will only remove the value when associated with that key, otherwise it will remove all instances of that value.
	End Rem
Function TreeRemoveValue(tree:BinTree,value:Object,key:String="")
	tree.removevalue value,key
End Function
	Rem
	bbdoc: Get the object associated with some key
	returns: The first value in the tree associated with the desired key, null if one does not exist
	End Rem
Function TreeFind:Object(tree:BinTree,key:String)
	Return tree.find(key)
End Function
	Rem
	bbdoc: Get the node associated with some key
	returns: The node in the tree with the specified key, null if none exists
	End Rem
Function TreeFindNode:BinNode(tree:BinTree,key:String)
	Return tree.findnode(key)
End Function
	Rem
	bbdoc: Get a list of all the objects associated with some key
	returns: A list containing all the values in the tree associated with the desired key
	End Rem
Function TreeFindAll:TList(tree:BinTree,key:String)
	Return tree.findall(key)
End Function
	Rem
	bbdoc: Get the number of the objects associated with some key
	returns: The number of values in the tree associated with the desired key
	End Rem
Function TreeFindCount:Int(tree:BinTree,key:String)
	Return tree.findcount(key)
End Function
	Rem
	bbdoc: Get an array for the contents of a tree
	returns: An array containing the objects in the tree
	about: @sort can be one of:
		[ @{Value} | @{Result}
		
		* 0 | Elements in the array are ordered via an in-order traversal
		
		* 1 | Elements in the array are ordered via a reversed in-order traversal
		
		* 2 | Elements in the array are ordered via a pre-order traversal
		
		* 3 | Elements in the array are ordered via a post-order traversal
		]
	End Rem
Function TreeToArray:Object[](tree:BinTree,sort:Int=0)
	Return tree.toarray(sort)
End Function
	Rem
	bbdoc: Optimize the tree so that it is as close to balanced as possible
	End Rem
Function OptimizeTree(tree:BinTree)
	tree.optimize
End Function
	Rem
	bbdoc: Returns an iterator object to that can be used with #EachIn
	returns: An iterator object
	about: Iterates through the objects in the tree
	End Rem
Function TreeObjects:BinEnum(tree:BinTree)
	Return tree.ObjectEnumerator()
End Function
	Rem
	bbdoc: Returns an iterator object to that can be used with #EachIn
	returns: An iterator object
	about: Iterates through the keys in the tree
	End Rem
Function TreeKeys:BinEnum(tree:BinTree)
	Return tree.KeyEnumerator()
End Function
	Rem
	bbdoc: Returns an iterator object to that can be used with #EachIn
	returns: An iterator object
	about: Iterates through the nodes in the tree
	End Rem
Function TreeNodes:BinEnum(tree:BinTree)
	Return tree.NodeEnumerator()
End Function
	Rem
	bbdoc: Copy a tree
	returns: A tree with identical key and value pairs but different node objects
	End Rem
Function CopyTree:BinTree(tree:BinTree)
	Return tree.copy()
End Function
	Rem
	bbdoc: Gets node contents as a list of comma separated values
	returns: A string containing objects cast as strings separated by commas
	End Rem
Function BinNodeString:String(node:BinNode)
	Return node.csv()
End Function
	Rem
	bbdoc: Gets node contents as a TList
	returns: A TList containing all the objects associated with this node's key
	End Rem
Function BinNodeContents:TList(node:BinNode)
	Return node.values()
End Function
	Rem
	bbdoc: Gets node key
	returns: The node's key
	End Rem
Function BinNodeKey:String(node:BinNode)
	Return node.key
End Function
	Rem
	bbdoc: Gets node value
	returns: The node's value
	about: Only the first value is returned, not any others with duplicate keys
	End Rem
Function BinNodeValue:Object(node:BinNode)
	Return node.value()
End Function


Type BinNodeData
	Field nxt:BinNodeData
	Field data:Object
	Function Create:BinNodeData(o:Object)
		Local n:BinNodeData=New BinNodeData
		n.data=o
		Return n
	End Function
	Method copy:BinNodeData()
		Local n:BinNodeData=New BinNodeData
		n.data=data
		If nxt Then n.nxt=nxt.copy()
		Return n
	End Method
	Method csv$()
		Local on:BinNodeData=Self,s$=String(on.data)
		Repeat
			on=on.nxt
			If Not on Exit
			s:+", "+String(on.data)
		Forever
		Return s
	End Method
End Type
Rem
bbdoc: A single node for a binary tree
End Rem
Type BinNode
	Field key$
	Field data:BinNodeData
	Field parent:BinNode
	Field l:BinNode,r:BinNode
	Rem
	bbdoc: Gets node contents as a list of comma separated values
	returns: A string containing objects cast as strings separated by commas
	End Rem
	Method csv$()
		If Not data Return ""
		Return data.csv()
	End Method
	Rem
	bbdoc: Iterates through the left children starting at this node until a node is found which does not have a left child, and returns that node.
	returns: The leftmost child of the node
	End Rem
	Method leftmostchild:BinNode()
		Local on:BinNode=Self
		While on.l
			on=on.l
		Wend
		Return on
	End Method
	Rem
	bbdoc: Iterates through the right children starting at this node until a node is found which does not have a right child, and returns that node.
	returns: The rightmost child of the node
	End Rem
	Method rightmostchild:BinNode()
		Local on:BinNode=Self
		While on.r
			on=on.r
		Wend
		Return on
	End Method
	Rem
	bbdoc: Move this node so that it becomes the rightmost child of the specified node
	End Rem
	Method movetoright(t:BinNode)
		Local n:BinNode=t.rightmostchild()
		orphan
		n.r=Self
		parent=n
	End Method
	Rem
	bbdoc: Move this node so that it becomes the leftmost child of the specified node
	End Rem
	Method movetoleft(t:BinNode)
		Local n:BinNode=t.leftmostchild()
		orphan
		n.l=Self
		parent=n
	End Method
	Rem
	bbdoc: Sever links to and from the node's parent
	End Rem
	Method orphan()
		If parent
			If parent.l=Self
				parent.l=Null
			ElseIf parent.r=Self
				parent.r=Null
			EndIf
			parent=Null
		EndIf
	End Method
	Rem
	bbdoc: Remove a node and return the node which replaces it, null if none does
	returns: Node with the new equivalent position as the removed one, null if none such node exists
	End Rem
	Method remove:BinNode()
		If l
			If parent Then
				l.parent=parent
				If parent.l=Self Then 
					parent.l=l
				ElseIf parent.r=Self
					parent.r=l
				EndIf
			EndIf
			If r Then r.movetoright(l.rightmostchild())
			Return l
		ElseIf r
			If parent Then 
				r.parent=parent
				If parent.l=Self Then 
					parent.l=r
				ElseIf parent.r=Self
					parent.r=r
				EndIf
			EndIf
			Return r
			'If l Then l.movetoleft(r.leftmostchild())
		EndIf
		orphan
		Return Null
	End Method
	Rem
	bbdoc: Remove a specific value from a node, and the node only if no values remain and @flag is true
	returns: The node which takes its place if this node gets removed, otherwise returns itself
	End Rem
	Method removevalue:BinNode(value:Object,flag%=True)
		Local i:BinNodeData=data,li:BinNodeData=Null
		While i
			If i.data=value Then 
				If li Then
					li.nxt=i.nxt
				Else
					data=i.nxt
				EndIf
			Else
				li=i
			EndIf
			i=i.nxt
		Wend
		'If data Then data=data.nxt
		If (Not data) And flag Then Return remove()
		Return Self
	End Method
	Rem
	bbdoc: Remove a single value from a node, and the node only if no values remain and @flag is true
	returns: The node which takes its place if this node gets removed, otherwise returns itself
	End Rem
	Method removeonevalue:BinNode(flag%=True)
		If data Then data=data.nxt
		If (Not data) And flag Then Return remove()
		Return Self
	End Method
	Rem
	bbdoc: Remove all the values from a node, And the node only If @flag is True
	returns: The node which takes its place if this node gets removed, otherwise returns itself
	End Rem
	Method removeallvalues:BinNode(flag%=False)
		data=Null
		If flag Then Return remove()
		Return Self
	End Method
	Rem
	bbdoc: Get the first value the node contains
	returns: An object
	End Rem
	Method value:Object()
		If Not data Return Null
		Return data.data
	End Method
	Rem
	bbdoc: Get all the values the node contains as a list
	returns: A #TList
	End Rem
	Method values:TList()
		Local r:TList=CreateList(),i:BinNodeData=data
		While i
			r.addlast i.data
			i=i.nxt
		Wend
		Return r
	End Method
	Rem
	bbdoc: Check whether a value belongs to the node.
	returns: True if it does, False otherwise.
	End Rem
	Method containsvalue:Int(val:Object)
		Local i:BinNodeData=data
		While i
			If i.data=val Return 1
			i=i.nxt
		Wend
		Return 0
	End Method
	Rem
	bbdoc: Get the number of values belonging to the node
	returns: The number of values
	End Rem
	Method valuecount%()
		Local r%=0,i:BinNodeData=data
		While i
			r:+1
			i=i.nxt
		Wend
		Return r
	End Method
	Rem
	bbdoc: Insertion algorithm
	End Rem
	Method insert:BinNode(n:BinNode)
		Local on:BinNode=Self
		Repeat
			If n.key>on.key
				If on.r
					on=on.r
				Else
					on.r=n;n.parent=on;Exit
				EndIf
			ElseIf n.key<on.key
				If on.l
					on=on.l
				Else
					on.l=n;n.parent=on;Exit
				EndIf
			Else
				on.addvalue n.data;Return on
			EndIf
		Forever
		Return n
	End Method
	Rem
	bbdoc: Add value(s) to this node, for handling duplicate keys
	about: Takes a BinNodeData for an argument, and appends all members of it onto the beginning of the node's existing list of values
	End Rem
	Method addvalue(n:BinNodeData)
		Local t:BinNodeData=data
		data=n
		If n
			While n.nxt
				n=n.nxt
			Wend
		EndIf
		n.nxt=t
	End Method
	Rem
	bbdoc: Recursive search algorithm
	returns: The first node in this node's subtree with the desired key, or the closest to the desired key
	End Rem
	Method findclosest:BinNode(k$)
		Local on:BinNode=Self
		Repeat
			If k=on.key Return on
			If k>on.key And on.r
				on=on.r
			ElseIf on.l
				on=on.l
			Else
				Return on
			EndIf
		Forever
	End Method
	Rem
	bbdoc: Recursive counting algorithm
	returns: The number of nodes in this node's subtree
	End Rem
	Method count%()
		Local sum%=1
		If l sum:+l.count()
		If r sum:+r.count()
		Return sum
	End Method
	Rem
	bbdoc: Recursive counting algorithm
	returns: The number of values in this node's subtree
	End Rem
	Method countvalues%()
		Local sum%=valuecount()
		If l sum:+l.countvalues()
		If r sum:+r.countvalues()
		Return sum
	End Method
	Rem
	bbdoc: Recursive counting algorithm
	returns: The height of this node's subtree
	about: A tree containing only its root is considered to have a height of 1
	End Rem
	Method height%()
		Local lh%=0,rh%=0
		If l lh=l.height()
		If r rh=r.height()
		Return Max(lh,rh)+1
	End Method
	Rem
	bbdoc: Recursive algorithm which discards all links between nodes in this node's subtree
	End Rem
	Method clearlinks()
		If l l.clearlinks;l=Null
		If r r.clearlinks;r=Null
		parent=Null
	End Method
	Rem
	bbdoc: Optimize this node's subtree so that it is as close to balanced as possible and return the new root
	returns: The root of the new, optimized tree
	End Rem
	Method optimize:BinNode() 'rearrange this entire tree with self as the root so that it's optimal and return the new root
		Local array:BinNode[]=nodesinorder(),p:BinNode=parent,lr%=0
		If parent
			If parent.l=Self lr=-1
			If parent.r=Self lr=1
			If lr=0 Return Self
		EndIf
		clearlinks
		Local newroot:BinNode=optimize_(0,array.length-1,array)
		If newroot
			newroot.parent=p
			If lr=-1 
				p.l=newroot
			ElseIf lr=1 
				p.r=newroot
			EndIf
		EndIf
		Return newroot
	End Method
	Function optimize_:BinNode(start%,fin%,array:BinNode[])
		If fin>=start
			Local middle%=Ceil((fin-start)/2.0)+start
			Local lf%=middle-1
			Local rs%=middle+1
			If lf>=start array[middle].l=optimize_(start,lf,array)
			If array[middle].l array[middle].l.parent=array[middle]
			If rs<=fin array[middle].r=optimize_(rs,fin,array)
			If array[middle].r array[middle].r.parent=array[middle]
			Return array[middle]
		EndIf
	End Function
	
	Rem
	bbdoc: Perform an in-order traversal of the tree
	returns: An array of nodes organized such that this node's subtree was traversed in-order
	End Rem
	Method nodesinorder:BinNode[]() 'return array of BinNodes resulting from an in-order traversal
		Local index%=0,array:BinNode[count()]
		nodesinorder_(index,array)
		Return array
	End Method
	Method nodesinorder_(index% Var,array:BinNode[])
		If l l.nodesinorder_(index,array)
		array[index]=Self;index:+1
		If r r.nodesinorder_(index,array)
	End Method
	Rem
	bbdoc: Perform a reversed in-order traversal of the tree
	returns: An array of nodes organized such that this node's subtree was traversed reverse in-order
	End Rem
	Method nodesreverseorder:BinNode[]()
		Local index%=0,array:BinNode[count()]
		nodesreverseorder_(index,array)
		Return array
	End Method
	Method nodesreverseorder_(index% Var,array:BinNode[])
		If r r.nodesreverseorder_(index,array)
		array[index]=Self;index:+1
		If l l.nodesreverseorder_(index,array)
	End Method
	Rem
	bbdoc: Perform a pre-order traversal of the tree
	returns: An array of nodes organized such that this node's subtree was traversed pre-order
	End Rem
	Method nodespreorder:BinNode[]()
		Local index%=0,array:BinNode[count()]
		nodespreorder_(index,array)
		Return array
	End Method
	Method nodespreorder_(index% Var,array:BinNode[])
		array[index]=Self;index:+1
		If l l.nodespreorder_(index,array)
		If r r.nodespreorder_(index,array)
	End Method
	Rem
	bbdoc: Perform a post-order traversal of the tree
	returns: An array of nodes organized such that this node's subtree was traversed post-order
	End Rem
	Method nodespostorder:BinNode[]()
		Local index%=0,array:BinNode[count()]
		nodespostorder_(index,array)
		Return array
	End Method
	Method nodespostorder_(index% Var,array:BinNode[])
		If l l.nodespostorder_(index,array)
		If r r.nodespostorder_(index,array)
		array[index]=Self;index:+1
	End Method
	
	Rem
	bbdoc: Perform an in-order traversal of the tree
	returns: An array of objects organized such that this node's subtree was traversed in-order
	End Rem
	Method inorder:Object[]() 'return array of BinNodes resulting from an in-order traversal
		Local index%=0,array:Object[count()]
		inorder_(index,array)
		Return array
	End Method
	Method inorder_(index% Var,array:Object[])
		If l l.inorder_(index,array)
		array[index]=data;index:+1
		If r r.inorder_(index,array)
	End Method
	Rem
	bbdoc: Perform a reversed in-order traversal of the tree
	returns: An array of objects organized such that this node's subtree was traversed reverse in-order
	End Rem
	Method reverseorder:Object[]()
		Local index%=0,array:Object[count()]
		reverseorder_(index,array)
		Return array
	End Method
	Method reverseorder_(index% Var,array:Object[])
		If r r.reverseorder_(index,array)
		array[index]=data;index:+1
		If l l.reverseorder_(index,array)
	End Method
	Rem
	bbdoc: Perform a pre-order traversal of the tree
	returns: An array of objects organized such that this node's subtree was traversed pre-order
	End Rem
	Method preorder:Object[]()
		Local index%=0,array:Object[count()]
		preorder_(index,array)
		Return array
	End Method
	Method preorder_(index% Var,array:Object[])
		array[index]=data;index:+1
		If l l.preorder_(index,array)
		If r r.preorder_(index,array)
	End Method
	Rem
	bbdoc: Perform a post-order traversal of the tree
	returns: An array of objects organized such that this node's subtree was traversed post-order
	End Rem
	Method postorder:Object[]()
		Local index%=0,array:Object[count()]
		postorder_(index,array)
		Return array
	End Method
	Method postorder_(index% Var,array:Object[])
		If l l.postorder_(index,array)
		If r r.postorder_(index,array)
		array[index]=data;index:+1
	End Method

	
	'Method render(x%,y%,w%,h%)
	'	DrawText key,x+w/2,y
	'	If l DrawLine x+w/2,y+10,x+w/4,y+h-10;l.render x,y+h,w/2,h
	'	If r DrawLine x+w/2,y+10,x+w-w/4,y+h-10;r.render x+w/2,y+h,w/2,h
	'End Method
	
	Rem
	bbdoc: Create a new node
	returns: A new node
	End Rem
	Function Create:BinNode(key$,o:Object)
		Local n:BinNode=New BinNode
		n.key=key;n.data=BinNodeData.Create(o)
		Return n
	End Function
	
	Rem
	bbdoc: If the node has a right child then return that node's leftmost child, otherwise return the first node possessing a right child found while travelling up its ancestry
	returns: The node in the tree with the next greatest key, null if none exists
	End Rem
	Method successor:BinNode()
		If r Return r.leftmostchild()
		Local p:BinNode=parent,n:BinNode=Self
		While p And n=p.r
			n=p;p=p.parent
		Wend
		Return p
	End Method
	Rem
	bbdoc: If the node has a left child then return that node's rightmost child, otherwise return the first node possessing a left child found while travelling up its ancestry
	returns: The node in the tree with the next least key, null if none exists
	End Rem
	Method predecessor:BinNode()
		If l Return l.rightmostchild()
		Local p:BinNode=parent,n:BinNode=Self
		While p And n=p.l
			n=p;p=p.parent
		Wend
		Return p
	End Method
	Rem
	bbdoc: Copy the subtree which this node is the root of
	returns: A subtree with identical key and value pairs but different node objects
	End Rem
	Method copy:BinNode()
		Local ret:BinNode=New BinNode
		ret.key=key
		ret.data=data.copy()
		If l Then
			ret.l=l.copy()
			ret.l.parent=ret
		EndIf
		If r Then
			ret.r=r.copy()
			ret.r.parent=ret
		EndIf
		Return ret
	End Method
End Type


Rem
bbdoc: A binary tree object
End Rem
Type BinTree
	Field root:BinNode
	Field autosplay%=1
	Rem
	bbdoc: Determine whether the tree is empty
	returns: 1 if the tree is empty, 0 if it contains at least a root
	End Rem
	Method isempty%()
		Return Not root
	End Method
	Rem
	bbdoc: Determine whether the tree is set to be a splay tree
	returns: 1 if the tree is set to splay automatically, 0 if not
	End Rem
	Method issplay%()
		Return autosplay
	End Method
	Rem
	bbdoc: Set whether the tree is set to be a splay tree
	End Rem
	Method setsplay%(dosplay%)
		autosplay=dosplay
	End Method
	Rem
	bbdoc: Copy the tree
	returns: A tree with identical key and value pairs but different node objects
	End Rem
	Method copy:BinTree()
		Local cop:BinTree=New BinTree
		If Not root Then Return cop
		cop.root=root.copy()
		Return cop
	End Method
	Rem
	bbdoc: Insert a new key and value pair into the tree
	returns: The newly inserted node
	End Rem
	Method insert:BinNode(key$,o:Object=Null,dosplay%=-1)
		If o=Null Then o=key
		Local n:BinNode=BinNode.Create(key,o)
		If root
			n=root.insert(n)
		Else
			root=n
		EndIf
		If dosplay=1 Or (dosplay=-1 And autosplay=1) splay n
		Return n
	End Method
	Rem
	bbdoc: Get the node with the closest key to some key
	returns: The first node in the tree with the desired key, or the node with a value closest to the desired key
	End Rem
	Method findclosestnode:BinNode(key$,dosplay%=-1)
		If Not root Return Null
		Local n:BinNode=root.findclosest(key)
		If dosplay=1 Or (dosplay=-1 And autosplay=1) splay n
		Return n
	End Method
	Rem
	bbdoc: Get the node with some key
	returns: The first node in the tree with the desired key, null if one does not exist
	End Rem
	Method findnode:BinNode(key$,dosplay%=-1)
		If Not root Return Null
		Local ret:BinNode=root.findclosest(key)
		If ret.key=key 
			If dosplay=1 Or (dosplay=-1 And autosplay=1) splay ret
			Return ret
		EndIf
		Return Null
	End Method
	Rem
	bbdoc: Get the object associated with the closest key to some key
	returns: The first value in the tree associated with the desired key, or the value associated with the key closest to the desired key
	End Rem
	Method findclosest:Object(key$,dosplay%=-1)
		Return findclosestnode(key,dosplay).value()
	End Method
	Rem
	bbdoc: Determine whether some key is present in the tree
	returns: 1 if there is any node in the tree with the specified key, 0 if there is not
	End Rem
	Method contains%(key$,dosplay%=0)
		If Not root Then Return 0
		If findclosestnode(key,dosplay).key=key Return 1
		Return 0
	End Method
	Rem
	bbdoc: Get the object associated with some key
	returns: The first value in the tree associated with the desired key, null if one does not exist
	End Rem
	Method find:Object(key$,dosplay%=-1)
		Local n:BinNode=findnode(key,dosplay)
		If Not n Then Return Null
		Return n.value()
	End Method
	Rem
	bbdoc: Get a list of all the objects associated with some key
	returns: A list containing all the values in the tree associated with the desired key
	End Rem
	Method findall:TList(key$,dosplay%=-1)
		If Not root Return CreateList()
		Local ret:BinNode=root.findclosest(key)
		If ret.key=key
			If dosplay=1 Or (dosplay=-1 And autosplay=1) splay ret
			Return ret.values()
		EndIf
		Return CreateList()
	End Method
	Rem
	bbdoc: Get the number of objects associated with some key
	returns: The number of the values in the tree associated with the desired key
	End Rem
	Method findcount%(key$,dosplay%=-1)
		If Not root Return 0
		Local ret:BinNode=root.findclosest(key)
		If ret.key=key
			If dosplay=1 Or (dosplay=-1 And autosplay=1) splay ret
			Return ret.valuecount()
		EndIf
		Return 0
	End Method
	Rem
	bbdoc: Remove the first associated object with the desired key
	returns: The node that the association was removed from, null if no node was affected
	End Rem
	Method remove:BinNode(key$)
		If Not root Return Null
		Local n:BinNode=root.findclosest(key)
		If n.key<>key Then Return Null
		If n=root root=n.removeonevalue() Else n.removeonevalue()
		Return n
	End Method
	Rem
	bbdoc: Remove the all associations with the desired key
	returns: The node that was removed
	End Rem
	Method removeall:BinNode(key$)
		If Not root Return Null
		Local n:BinNode=root.findclosest(key)
		If n.key<>key Then Return Null
		If n=root root=n.remove() Else n.remove()
		Return n
	End Method
	Rem
	bbdoc: Remove a node from the tree
	returns: The node which took its new position, if any
	End Rem
	Method removenode:BinNode(n:BinNode)
		If n=root root=n.remove() Else Return n.remove()
		Return root
	End Method
	Rem
	bbdoc: Remove a specific value from the tree. If a key the value is associated with not specified then it will search through the entire tree and remove all instances, otherwise it will remove the value only from the node for that key.
	End Rem
	Method removevalue(value:Object,key$="")
		If key
			Local node:BinNode=findnode(key,False)
			If Not node Return
			If node=root Then root=node.removevalue(value) Else node.removevalue(value)
		Else
			For Local node:BinNode=EachIn NodeEnumerator()
				If node=root Then root=node.removevalue(value) Else node.removevalue(value)
			Next
		EndIf
	End Method
	Rem
	bbdoc: Count the number of nodes
	returns: The number of nodes in the tree
	End Rem
	Method count%()
		If root Return root.count()
		Return 0
	End Method
	Rem
	bbdoc: Count the number of values
	returns: The number of values in the tree
	End Rem
	Method countvalues%()
		If root Return root.countvalues()
		Return 0
	End Method
	Rem
	bbdoc: A tree with no nodes is considered to have a height of 0, and a tree with only a root to have a height of 1
	returns: The height of the tree
	End Rem
	Method height%()
		If root Return root.height()
		Return 0
	End Method
	Rem
	bbdoc: Clear all contents of the tree
	End Rem
	Method clear()
		If root Then root.clearlinks
		root=Null
	End Method
	Rem
	bbdoc: Optimize the tree so that it is as close to balanced as possible
	End Rem
	Method optimize()
		If root root=root.optimize()
	End Method
	Rem
	returns: An array containing the objects in the tree
	about: @sort can be one of:
		[ @{Value} | @{Result}
		
		* 0 | Elements in the array are ordered via an in-order traversal
		
		* 1 | Elements in the array are ordered via a reversed in-order traversal
		
		* 2 | Elements in the array are ordered via a pre-order traversal
		
		* 3 | Elements in the array are ordered via a post-order traversal
		]
	End Rem
	Method toarray:Object[](sort%=0)
		If root 
			If sort=0 
				Return root.inorder()
			ElseIf sort=1
				Return root.reverseorder()
			ElseIf sort=2
				Return root.preorder()
			ElseIf sort=3
				Return root.postorder()
			EndIf
		EndIf
		Return Null
	End Method
	'Method render(x%,y%,w%,h%)
	'	If root root.render(x,y,w,h/root.height())
	'End Method
	Rem
	bbdoc: Create a new tree
	returns: A new tree
	End Rem
	Function Create:BinTree(splay%=True)
		Local t:BinTree=New BinTree
		t.autosplay=splay
		Return t
	End Function
	Rem
	bbdoc: Method implements #EachIn support
	returns: An iterator object
	about: Iterates through the objects in the tree
	End Rem
	Method ObjectEnumerator:BinEnum()
		If Not root Return New BinEnum
		Return BinEnum.Create(root.leftmostchild(),0)
	End Method
	Rem
	bbdoc: Method implements #EachIn support
	returns: An iterator object
	about: Iterates through the keys in the tree
	End Rem
	Method KeyEnumerator:BinEnum()
		If Not root Return New BinEnum
		Return BinEnum.Create(root.leftmostchild(),1)
	End Method
	Rem
	bbdoc: Method implements #EachIn support
	returns: An iterator object
	about: Iterates through the nodes in the tree
	End Rem
	Method NodeEnumerator:BinEnum()
		If Not root Return New BinEnum
		Return BinEnum.Create(root.leftmostchild(),2)
	End Method
	Rem
	bbdoc: Splay a node such that it becomes the new root of the tree
	End Rem
	Method splay(n:BinNode)
		If n=root Or n.parent=Null Then Return
		If n.parent=root Then
			If n=n.parent.l Then
				rotright(root)
			Else
				rotleft(root)
			EndIf
		Else
			Local p:BinNode=n.parent,g:BinNode=p.parent
			If n=p.l And p=g.l
				rotright(g)
				rotright(p)
			ElseIf n=p.r And p=g.r
				rotleft(g)
				rotleft(p)
			ElseIf n=p.r And p=g.l
				rotleft(p)
				rotright(g)
			ElseIf n=p.l And p=g.r
				rotright(p)
				rotleft(g)
			EndIf
			splay n
		EndIf
	End Method
	Rem
	bbdoc: Node rotation right
	End Rem
	Method rotright(n:BinNode)
		Local x:BinNode=n.l
		n.l=x.r
		If x.r Then x.r.parent=n
		x.r=n
		If n.parent
			If n=n.parent.r Then n.parent.r=x Else n.parent.l=x
		EndIf
		x.parent=n.parent;n.parent=x
		If n=root root=x
	End Method
	Rem
	bbdoc: Node rotation left
	End Rem
	Method rotleft(n:BinNode)
		Local x:BinNode=n.r
		n.r=x.l
		If x.l Then x.l.parent=n
		x.l=n
		If n.parent
			If n=n.parent.l Then n.parent.l=x Else n.parent.r=x
		EndIf
		x.parent=n.parent;n.parent=x
		If n=root Then root=x
	End Method
End Type

Type BinEnum
	Field node:BinNode,data:BinNodeData,kind%=0
	Function Create:BinEnum(n:BinNode,kind%=0)
		Local e:BinEnum=New BinEnum
		e.node=n
		e.data=n.data
		e.kind=kind
		Return e
	End Function
	Method HasNext%()
		If Not kind
			If data Return 1
			While node And Not data
				node=node.successor()
				If node Then data=node.data
			Wend
			If Not data Return 0
			Return 1
		EndIf
		Return node<>Null
	End Method
	Method NextObject:Object()
		Local n:BinNode=node
		If kind Then	node=node.successor()
		If kind=1
			Return n.key
		ElseIf kind=2
			Return n
		Else
			Local ret:Object=data.data
			data=data.nxt
			Return ret
		EndIf
	End Method
	Method ObjectEnumerator:BinEnum()
		Return Self
	End Method
End Type












